import { NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"
import { createRepertoireGroupSchema, userIdQuerySchema, validateData, validateQueryData } from "@/lib/validations"

const sql = neon(process.env.DATABASE_URL!)

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const userIdParam = searchParams.get("userId")

    if (!userIdParam) {
      return NextResponse.json(
        { error: "userId é obrigatório" },
        { status: 400 }
      )
    }

    // Validar parâmetro userId
     try {
       const { userId } = userIdQuerySchema.parse({ userId: userIdParam })
       
       // Get repertoire groups with songs
       const groups = await sql`
         SELECT 
           rg.id,
           rg.name,
           rg.description,
           rg.color,
           rg.position,
           rg.created_at,
           rg.updated_at,
           COALESCE(
             JSON_AGG(
               JSON_BUILD_OBJECT(
                 'id', s.id,
                 'title', s.title,
                 'artist', s.artist,
                 'youtube_url', s.youtube_url,
                 'youtube_video_id', s.youtube_video_id,
                 'duration', s.duration,
                 'position', s.position,
                 'created_at', s.created_at,
                 'updated_at', s.updated_at
               ) ORDER BY s.position ASC
             ) FILTER (WHERE s.id IS NOT NULL),
             '[]'::json
           ) as songs
         FROM repertoire_groups rg
         LEFT JOIN songs s ON rg.id = s.repertoire_group_id
         WHERE rg.user_id = ${userId}
         GROUP BY rg.id, rg.name, rg.description, rg.color, rg.position, rg.created_at, rg.updated_at
         ORDER BY rg.position ASC
       `

       return NextResponse.json({
         success: true,
         groups: groups,
       })
     } catch (error) {
       if (error instanceof Error && error.name === 'ZodError') {
         return NextResponse.json(
           { 
             error: "Parâmetro inválido", 
             details: error.message 
           },
           { status: 400 }
         )
       }
       
       console.error("[v0] Error fetching repertoire groups:", error)
        return NextResponse.json(
          { error: "Erro interno do servidor" },
          { status: 500 }
        )
      }
    } catch (outerError) {
      console.error("[v0] Outer error:", outerError)
      return NextResponse.json(
        { error: "Erro interno do servidor" },
        { status: 500 }
      )
    }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    
    // Validar dados de entrada
    const validation = validateData(createRepertoireGroupSchema, body)
    if (!validation.success) {
      return NextResponse.json(
        { 
          error: "Dados inválidos", 
          details: validation.errors 
        },
        { status: 400 }
      )
    }

    const { name, description, color, userId } = validation.data

    // Get next position
    const positionResult = await sql`
      SELECT COALESCE(MAX(position), 0) + 1 as next_position
      FROM repertoire_groups
      WHERE user_id = ${userId}
    `

    const nextPosition = positionResult[0].next_position

    const result = await sql`
      INSERT INTO repertoire_groups (name, description, color, position, user_id)
      VALUES (${name}, ${description || ""}, ${color || "#3b82f6"}, ${nextPosition}, ${userId})
      RETURNING id, name, description, color, position, user_id, created_at
    `

    return NextResponse.json({
      success: true,
      group: { ...result[0], songs: [] },
    })
  } catch (error) {
    console.error("Create repertoire error:", error)
    return NextResponse.json(
      {
        success: false,
        message: "Erro interno do servidor",
      },
      { status: 500 },
    )
  }
}
