"use client"

import React from "react"
import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Label } from "@/components/ui/label"
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select"
import { Users, Plus, LogOut, Music, Edit, Search, Filter, ShoppingCart, FolderPlus, Settings } from "lucide-react"
import { useRouter } from "next/navigation"
import MusicStoreModal from "@/components/music-store-modal"
import AddMusicModal from "@/components/add-music-modal"
import CategoryManagementModal from "@/components/category-management-modal"
import { authClient } from "@/lib/auth-client"

interface User {
  id: number
  name: string
  email: string
  status: "active" | "inactive"
  created_at: string
}

interface EditUser {
  id: number
  name: string
  email: string
}

export default function MasterDashboard() {
  const [users, setUsers] = useState<User[]>([])
  const [filteredUsers, setFilteredUsers] = useState<User[]>([])
  const [newUser, setNewUser] = useState({ name: "", email: "", password: "" })
  const [editUser, setEditUser] = useState<EditUser | null>(null)
  const [isDialogOpen, setIsDialogOpen] = useState(false)
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [isLoading, setIsLoading] = useState(false)
  const [isAuthenticated, setIsAuthenticated] = useState(false)
  const [currentUser, setCurrentUser] = useState<any>(null)
  
  // Estados dos filtros
  const [filters, setFilters] = useState({
    email: "",
    name: "",
    status: "all" as "all" | "active" | "inactive"
  })
  const [isMusicStoreOpen, setIsMusicStoreOpen] = useState(false)
  const [isAddMusicOpen, setIsAddMusicOpen] = useState(false)
  const [isCategoryManagementOpen, setIsCategoryManagementOpen] = useState(false)
  
  const router = useRouter()

  useEffect(() => {
    checkAuth()
  }, [])

  const checkAuth = async () => {
    try {
      console.log("[v0] Dashboard checking auth...")

      const localAuth = authClient.getAuth()
      if (localAuth) {
        console.log("[v0] Found local auth data, verifying with server")

        const response = await fetch("/api/auth/check", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ authData: localAuth }),
        })

        const data = await response.json()

        if (data.authenticated && data.user.type === "master") {
          console.log("[v0] Hybrid auth successful")
          setIsAuthenticated(true)
          setCurrentUser(data.user)
          loadUsers()
          return
        }
      }

      // Fallback para verificação tradicional com cookies
      const response = await fetch("/api/auth/check")
      const data = await response.json()

      console.log("[v0] Auth check response:", data)

      if (!data.authenticated || data.user.type !== "master") {
        console.log("[v0] Not authenticated or not master, redirecting to login")
        authClient.clearAuth() // Limpa dados locais inválidos
        router.push("/")
        return
      }

      console.log("[v0] Auth successful, setting state")
      setIsAuthenticated(true)
      setCurrentUser(data.user)
      loadUsers()
    } catch (error) {
      console.error("Auth check error:", error)
      console.log("[v0] Auth check failed, redirecting to login")
      authClient.clearAuth()
      router.push("/")
    }
  }

  const loadUsers = async () => {
    try {
      const response = await fetch("/api/users")
      const data = await response.json()

      if (data.success) {
        // Convert is_active boolean to status string for UI compatibility
        const usersWithStatus = data.users.map((user: any) => ({
          ...user,
          status: user.is_active ? "active" : "inactive"
        }))
        setUsers(usersWithStatus)
        setFilteredUsers(usersWithStatus)
      }
    } catch (error) {
      console.error("Error loading users:", error)
    }
  }

  // Função para aplicar filtros
  const applyFilters = () => {
    let filtered = users

    // Filtro por email
    if (filters.email) {
      filtered = filtered.filter(user => 
        user.email.toLowerCase().includes(filters.email.toLowerCase())
      )
    }

    // Filtro por nome
    if (filters.name) {
      filtered = filtered.filter(user => 
        user.name.toLowerCase().includes(filters.name.toLowerCase())
      )
    }

    // Filtro por status
    if (filters.status !== "all") {
      filtered = filtered.filter(user => user.status === filters.status)
    }

    setFilteredUsers(filtered)
  }

  // Aplicar filtros sempre que os filtros ou usuários mudarem
  useEffect(() => {
    applyFilters()
  }, [filters, users])

  // Limpar filtros
  const clearFilters = () => {
    setFilters({
      email: "",
      name: "",
      status: "all"
    })
  }

  const handleLogout = async () => {
    try {
      authClient.clearAuth()
      await fetch("/api/auth/logout", { method: "POST" })
      window.location.href = "/"
    } catch (error) {
      console.error("Logout error:", error)
      authClient.clearAuth()
      window.location.href = "/"
    }
  }

  const toggleUserStatus = async (userId: number) => {
    const user = users.find((u) => u.id === userId)
    if (!user) return

    const newStatus = user.status === "active" ? "inactive" : "active"

    try {
      const response = await fetch(`/api/users/${userId}`, {
        method: "PATCH",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ status: newStatus }),
      })

      const data = await response.json()

      if (data.success) {
        setUsers(users.map((u) => (u.id === userId ? { ...u, status: newStatus } : u)))
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error updating user status:", error)
      alert("Erro ao atualizar status do usuário")
    }
  }

  const handleAddUser = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)

    try {
      const response = await fetch("/api/users", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(newUser),
      })

      const data = await response.json()

      if (data.success) {
        // Convert is_active to status for UI compatibility
        const userWithStatus = {
          ...data.user,
          status: data.user.is_active ? "active" : "inactive"
        }
        setUsers([...users, userWithStatus])
        setNewUser({ name: "", email: "", password: "" })
        setIsDialogOpen(false)
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error creating user:", error)
      alert("Erro ao criar usuário")
    }

    setIsLoading(false)
  }

  const handleEditUser = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!editUser) return
    
    setIsLoading(true)

    try {
      const response = await fetch(`/api/users/${editUser.id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          name: editUser.name,
          email: editUser.email,
        }),
      })

      const data = await response.json()

      if (data.success) {
        // Update user in the list
        setUsers(users.map(user => 
          user.id === editUser.id 
            ? { ...user, name: editUser.name, email: editUser.email }
            : user
        ))
        setEditUser(null)
        setIsEditDialogOpen(false)
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error updating user:", error)
      alert("Erro ao atualizar usuário")
    }

    setIsLoading(false)
  }

  const openEditDialog = (user: User) => {
    setEditUser({
      id: user.id,
      name: user.name,
      email: user.email,
    })
    setIsEditDialogOpen(true)
  }

  if (!isAuthenticated) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <Music className="h-12 w-12 text-purple-600 mx-auto mb-4" />
          <p className="text-gray-600">Verificando autenticação...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Music className="h-8 w-8 text-purple-600 mr-3" />
              <h1 className="text-xl font-semibold text-gray-900">Dashboard Master</h1>
            </div>
            <div className="flex items-center space-x-3">
              <Button 
                 onClick={() => router.push('/music-store')} 
                 variant="outline" 
                 size="sm"
               >
                 <ShoppingCart className="h-4 w-4 mr-2" />
                 Loja de Músicas
               </Button>
              <Button 
                 onClick={() => setIsAddMusicOpen(true)} 
                 variant="outline" 
                 size="sm"
               >
                 <FolderPlus className="h-4 w-4 mr-2" />
                 Adicionar Música
               </Button>
              <Button 
                 onClick={() => setIsCategoryManagementOpen(true)} 
                 variant="outline" 
                 size="sm"
               >
                 <Settings className="h-4 w-4 mr-2" />
                 Categorias
               </Button>
              <Button onClick={handleLogout} variant="outline" size="sm">
                <LogOut className="h-4 w-4 mr-2" />
                Sair
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="mb-8">
          {/* Seção de Filtros */}
          <div className="bg-white p-6 rounded-lg shadow-sm border mb-6">
            <div className="flex items-center space-x-2 mb-4">
              <Filter className="h-5 w-5" />
              <h2 className="text-lg font-semibold">Filtros de Busca</h2>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              {/* Filtro por Email */}
              <div>
                <Label htmlFor="filter-email">Email</Label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    id="filter-email"
                    type="email"
                    placeholder="Buscar por email..."
                    value={filters.email}
                    onChange={(e) => setFilters(prev => ({ ...prev, email: e.target.value }))}
                    className="pl-10"
                  />
                </div>
              </div>

              {/* Filtro por Nome */}
              <div>
                <Label htmlFor="filter-name">Nome</Label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    id="filter-name"
                    type="text"
                    placeholder="Buscar por nome..."
                    value={filters.name}
                    onChange={(e) => setFilters(prev => ({ ...prev, name: e.target.value }))}
                    className="pl-10"
                  />
                </div>
              </div>

              {/* Filtro por Status */}
              <div>
                <Label htmlFor="filter-status">Status</Label>
                <Select
                  value={filters.status}
                  onValueChange={(value) => setFilters(prev => ({ ...prev, status: value as "all" | "active" | "inactive" }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecionar status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    <SelectItem value="active">Ativo</SelectItem>
                    <SelectItem value="inactive">Inativo</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Botão Limpar Filtros */}
              <div className="flex items-end">
                <Button 
                  variant="outline" 
                  onClick={clearFilters}
                  className="w-full"
                >
                  Limpar Filtros
                </Button>
              </div>
            </div>

            {/* Contador de resultados */}
            <div className="mt-4 text-sm text-gray-600">
              Mostrando {filteredUsers.length} de {users.length} usuários
            </div>
          </div>

          <div className="flex justify-between items-center mb-6">
            <div>
              <h2 className="text-2xl font-bold text-gray-900">Gerenciar Usuários</h2>
              <p className="text-gray-600">Controle os usuários cadastrados no sistema</p>
            </div>

            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button>
                  <Plus className="h-4 w-4 mr-2" />
                  Adicionar Usuário
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle>Cadastrar Novo Usuário</DialogTitle>
                  <DialogDescription>Adicione um novo usuário ao sistema</DialogDescription>
                </DialogHeader>
                <form onSubmit={handleAddUser} className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="user-name">Nome</Label>
                    <Input
                      id="user-name"
                      value={newUser.name}
                      onChange={(e) => setNewUser({ ...newUser, name: e.target.value })}
                      placeholder="Nome completo"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="user-email">Email</Label>
                    <Input
                      id="user-email"
                      type="email"
                      value={newUser.email}
                      onChange={(e) => setNewUser({ ...newUser, email: e.target.value })}
                      placeholder="email@exemplo.com"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="user-password">Senha</Label>
                    <Input
                      id="user-password"
                      type="password"
                      value={newUser.password}
                      onChange={(e) => setNewUser({ ...newUser, password: e.target.value })}
                      placeholder="Senha do usuário"
                      required
                    />
                  </div>
                  <Button type="submit" className="w-full" disabled={isLoading}>
                    {isLoading ? "Cadastrando..." : "Cadastrar Usuário"}
                  </Button>
                </form>
              </DialogContent>
            </Dialog>
          </div>

          {/* Edit User Dialog */}
          <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Editar Usuário</DialogTitle>
                <DialogDescription>Altere as informações do usuário</DialogDescription>
              </DialogHeader>
              <form onSubmit={handleEditUser} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="edit-user-name">Nome</Label>
                  <Input
                    id="edit-user-name"
                    value={editUser?.name || ""}
                    onChange={(e) => setEditUser(editUser ? { ...editUser, name: e.target.value } : null)}
                    placeholder="Nome completo"
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit-user-email">Email</Label>
                  <Input
                    id="edit-user-email"
                    type="email"
                    value={editUser?.email || ""}
                    onChange={(e) => setEditUser(editUser ? { ...editUser, email: e.target.value } : null)}
                    placeholder="email@exemplo.com"
                    required
                  />
                </div>
                <div className="flex space-x-2">
                  <Button type="submit" className="flex-1" disabled={isLoading}>
                    {isLoading ? "Salvando..." : "Salvar Alterações"}
                  </Button>
                  <Button 
                    type="button" 
                    variant="outline" 
                    onClick={() => setIsEditDialogOpen(false)}
                    disabled={isLoading}
                  >
                    Cancelar
                  </Button>
                </div>
              </form>
            </DialogContent>
          </Dialog>

          {/* Users List */}
          <div className="grid gap-4">
            {filteredUsers.length === 0 ? (
              <Card>
                <CardContent className="flex items-center justify-center py-12">
                  <div className="text-center">
                    <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 mb-2">
                      {users.length === 0 ? "Nenhum usuário cadastrado" : "Nenhum usuário encontrado"}
                    </h3>
                    <p className="text-gray-500">
                      {users.length === 0 ? "Adicione o primeiro usuário para começar" : "Tente ajustar os filtros de busca"}
                    </p>
                  </div>
                </CardContent>
              </Card>
            ) : (
              filteredUsers.map((user) => (
                <Card key={user.id}>
                  <CardContent className="flex items-center justify-between p-6">
                    <div className="flex items-center space-x-4">
                      <div className="bg-purple-100 p-2 rounded-full">
                        <Users className="h-5 w-5 text-purple-600" />
                      </div>
                      <div>
                        <h3 className="font-medium text-gray-900">{user.name}</h3>
                        <p className="text-sm text-gray-500">{user.email}</p>
                        <p className="text-xs text-gray-400">
                          Cadastrado em {new Date(user.created_at).toLocaleDateString("pt-BR")}
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-3">
                      <Badge variant={user.status === "active" ? "default" : "secondary"}>
                        {user.status === "active" ? "Ativo" : "Inativo"}
                      </Badge>
                      <Button onClick={() => openEditDialog(user)} variant="outline" size="sm">
                        <Edit className="h-4 w-4 mr-1" />
                        Editar
                      </Button>
                      <Button onClick={() => toggleUserStatus(user.id)} variant="outline" size="sm">
                        {user.status === "active" ? "Desativar" : "Ativar"}
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))
            )}
          </div>
        </div>
      </main>

      <MusicStoreModal
        isOpen={isMusicStoreOpen}
        onClose={() => setIsMusicStoreOpen(false)}
        onAddToRepertoire={async (songId: number, groupId: number, position: number) => {
          // Para o master dashboard, podemos mostrar uma mensagem ou redirecionar
          // já que o master não tem repertórios próprios
          alert('Funcionalidade disponível apenas para usuários com repertórios')
        }}
      />

      <AddMusicModal
        isOpen={isAddMusicOpen}
        onClose={() => setIsAddMusicOpen(false)}
        onMusicAdded={() => {
          // Recarregar a loja de músicas se estiver aberta
          // ou mostrar mensagem de sucesso
          alert('Música adicionada com sucesso à loja!')
        }}
      />

      <CategoryManagementModal
        isOpen={isCategoryManagementOpen}
        onClose={() => setIsCategoryManagementOpen(false)}
      />
    </div>
  )
}
