'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Music, Search, Plus, Filter, ShoppingCart, Heart } from 'lucide-react';

interface Song {
  id: number;
  title: string;
  artist: string;
  youtube_url?: string;
  youtube_video_id?: string;
  duration?: string;
  category_id?: number;
  category_name?: string;
  source?: string;
  created_at: string;
}

interface Category {
  id: number;
  name: string;
  description: string;
  color: string;
}

interface RepertoireGroup {
  id: number;
  name: string;
}

export default function MusicStore() {
  const [songs, setSongs] = useState<Song[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [repertoireGroups, setRepertoireGroups] = useState<RepertoireGroup[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [currentUser, setCurrentUser] = useState<any>(null);

  // Estados para adicionar música ao repertório
  const [selectedSong, setSelectedSong] = useState<Song | null>(null);
  const [selectedGroup, setSelectedGroup] = useState<string>('');
  const [addingToRepertoire, setAddingToRepertoire] = useState(false);

  useEffect(() => {
    checkAuth();
    loadSongs();
    loadCategories();
  }, []);

  useEffect(() => {
    if (isAuthenticated && currentUser?.id) {
      loadRepertoireGroups();
    }
  }, [isAuthenticated, currentUser]);

  const checkAuth = async () => {
    try {
      const response = await fetch('/api/auth/check');
      const data = await response.json();
      setIsAuthenticated(data.authenticated);
      setCurrentUser(data.user);
    } catch (error) {
      console.error('Erro ao verificar autenticação:', error);
    }
  };

  const loadSongs = async () => {
    try {
      const response = await fetch('/api/music-store');
      const data = await response.json();
      if (Array.isArray(data)) {
        setSongs(data);
      }
    } catch (error) {
      console.error('Erro ao carregar músicas:', error);
    } finally {
      setLoading(false);
    }
  };

  const loadCategories = async () => {
    try {
      const response = await fetch('/api/categories');
      const data = await response.json();
      if (data.success) {
        setCategories(data.categories);
      }
    } catch (error) {
      console.error('Erro ao carregar categorias:', error);
    }
  };

  const loadRepertoireGroups = async () => {
    if (!currentUser?.id) {
      console.error('Usuário não encontrado para carregar grupos de repertório');
      return;
    }
    
    try {
      console.log('Carregando grupos para usuário:', currentUser.id);
      const response = await fetch(`/api/repertoires?userId=${currentUser.id}`);
      const data = await response.json();
      console.log('Resposta da API repertoires:', data);
      
      if (data.success) {
        setRepertoireGroups(data.groups);
        console.log('Grupos carregados:', data.groups);
      } else {
        console.error('Erro na resposta da API:', data.error);
      }
    } catch (error) {
      console.error('Erro ao carregar grupos de repertório:', error);
    }
  };

  const addToRepertoire = async () => {
    if (!selectedSong || !selectedGroup) {
      alert('Por favor, selecione uma música e um grupo de repertório.');
      return;
    }

    const groupId = parseInt(selectedGroup);
    if (isNaN(groupId)) {
      alert('Grupo de repertório inválido.');
      return;
    }

    const payload = {
      title: selectedSong.title,
      artist: selectedSong.artist,
      youtubeUrl: (selectedSong.youtube_url || '').trim().replace(/`/g, ''), // Removendo espaços extras e crases
      repertoireGroupId: groupId,
    };

    console.log('Payload sendo enviado:', payload); // Debug log
    console.log('selectedGroup:', selectedGroup, 'tipo:', typeof selectedGroup); // Debug log

    setAddingToRepertoire(true);
    try {
      const response = await fetch('/api/songs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(payload),
      });

      const data = await response.json();
      console.log('Resposta da API:', data); // Debug log
      console.log('Detalhes dos erros:', data.details); // Debug log específico para os detalhes
      
      if (response.ok && data.success) {
        alert('Música adicionada ao repertório com sucesso!');
        setSelectedSong(null);
        setSelectedGroup('');
      } else {
        console.error('Erro detalhado:', data);
        alert('Erro ao adicionar música ao repertório: ' + (data.error || data.message || 'Dados inválidos'));
      }
    } catch (error) {
      console.error('Erro ao adicionar música ao repertório:', error);
      alert('Erro ao adicionar música ao repertório: ' + error.message);
    } finally {
      setAddingToRepertoire(false);
    }
  };

  const filteredSongs = songs.filter(song => {
    const matchesSearch = song.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         song.artist.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || song.category_id?.toString() === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  // Filtrar duplicatas por URL do YouTube
  const uniqueSongs = filteredSongs.reduce((acc, song) => {
    const existingSong = acc.find(s => s.youtube_url === song.youtube_url);
    if (!existingSong) {
      acc.push(song);
    }
    return acc;
  }, [] as Song[]);

  // Verificação de autenticação obrigatória
  if (!loading && !isAuthenticated) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <ShoppingCart className="h-12 w-12 text-red-600 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Acesso Restrito</h2>
          <p className="text-gray-600 mb-4">Você precisa estar logado para acessar a loja de músicas.</p>
          <Button onClick={() => window.location.href = '/'}>
            Fazer Login
          </Button>
        </div>
      </div>
    );
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <Music className="h-12 w-12 text-blue-600 mx-auto mb-4 animate-pulse" />
          <p className="text-gray-600">Carregando loja de músicas...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center space-x-3">
              <ShoppingCart className="h-8 w-8 text-blue-600" />
              <h1 className="text-2xl font-bold text-gray-900">Loja de Músicas</h1>
            </div>
            <div className="flex items-center space-x-4">
              <Badge variant="outline" className="text-sm">
                {filteredSongs.length} músicas disponíveis
              </Badge>
              {isAuthenticated ? (
                <Button onClick={() => window.location.href = '/user-dashboard'}>
                  Meu Repertório
                </Button>
              ) : (
                <Button onClick={() => window.location.href = '/'}>
                  Fazer Login
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Filtros */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                <Input
                  placeholder="Buscar por título ou artista..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <div className="md:w-64">
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger>
                  <Filter className="h-4 w-4 mr-2" />
                  <SelectValue placeholder="Filtrar por categoria" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas as categorias</SelectItem>
                  {categories.map((category) => (
                    <SelectItem key={category.id} value={category.id.toString()}>
                      <div className="flex items-center space-x-2">
                        <div 
                          className="w-3 h-3 rounded-full" 
                          style={{ backgroundColor: category.color }}
                        />
                        <span>{category.name}</span>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>

        {/* Grid de Músicas */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {uniqueSongs.map((song) => (
            <Card key={song.id} className="overflow-hidden hover:shadow-lg transition-shadow">
              {/* Capa do Vídeo */}
              {song.youtube_video_id && (
                <div className="relative w-full h-48 bg-gray-200">
                  <img 
                    src={`https://img.youtube.com/vi/${song.youtube_video_id}/maxresdefault.jpg`}
                    alt={`Capa do vídeo: ${song.title}`}
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      // Fallback para thumbnail de qualidade menor se maxres não existir
                      e.currentTarget.src = `https://img.youtube.com/vi/${song.youtube_video_id}/hqdefault.jpg`;
                    }}
                  />
                  <div className="absolute inset-0 bg-black bg-opacity-20 flex items-center justify-center opacity-0 hover:opacity-100 transition-opacity">
                    <div className="bg-white bg-opacity-90 rounded-full p-3">
                      <Music className="h-6 w-6 text-gray-700" />
                    </div>
                  </div>
                </div>
              )}
              
              <div className="p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h3 className="font-semibold text-lg text-gray-900 mb-1">
                      {song.title}
                    </h3>
                    <p className="text-gray-600 mb-2">{song.artist}</p>
                    <div className="flex flex-wrap gap-2 mb-2">
                      {song.category_name && (
                        <Badge 
                          variant="outline" 
                          className="text-xs"
                        >
                          {song.category_name}
                        </Badge>
                      )}
                      {song.source && (
                        <Badge 
                          variant={song.source === 'Loja' ? 'default' : 'secondary'}
                          className="text-xs"
                        >
                          {song.source}
                        </Badge>
                      )}
                    </div>
                  </div>
                  {!song.youtube_video_id && (
                    <Music className="h-6 w-6 text-gray-400" />
                  )}
                </div>

              <div className="space-y-2 mb-4">
                {song.duration && (
                  <div className="flex justify-between text-sm text-gray-600">
                    <span>Duração:</span>
                    <span className="font-medium">{song.duration}</span>
                  </div>
                )}
                {song.youtube_url && (
                  <div className="flex justify-between text-sm text-gray-600">
                    <span>YouTube:</span>
                    <a 
                      href={song.youtube_url} 
                      target="_blank" 
                      rel="noopener noreferrer"
                      className="font-medium text-blue-600 hover:text-blue-800"
                    >
                      Ver vídeo
                    </a>
                  </div>
                )}
              </div>

              <div className="flex space-x-2">
                {song.youtube_url && (
                  <Button 
                    variant="outline" 
                    size="sm" 
                    className="flex-1"
                    onClick={() => window.open(song.youtube_url, '_blank')}
                  >
                    Ver no YouTube
                  </Button>
                )}

                {isAuthenticated && (
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button 
                        size="sm" 
                        className="flex-1"
                        onClick={() => setSelectedSong(song)}
                      >
                        <Plus className="h-4 w-4 mr-1" />
                        Adicionar
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>Adicionar ao Repertório</DialogTitle>
                        <DialogDescription>
                          Selecione um grupo de repertório para adicionar esta música.
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4 mt-4">
                        <div>
                          <Label>Música Selecionada</Label>
                          <p className="text-sm text-gray-600 mt-1">
                            {selectedSong?.title} - {selectedSong?.artist}
                          </p>
                        </div>
                        
                        <div>
                          <Label htmlFor="group">Grupo de Repertório</Label>
                          <Select value={selectedGroup} onValueChange={setSelectedGroup}>
                            <SelectTrigger>
                              <SelectValue placeholder="Selecione um grupo" />
                            </SelectTrigger>
                            <SelectContent>
                              {repertoireGroups.length > 0 ? (
                                repertoireGroups.map((group) => (
                                  <SelectItem key={group.id} value={group.id.toString()}>
                                    {group.name}
                                  </SelectItem>
                                ))
                              ) : (
                                <SelectItem value="" disabled>
                                  Nenhum grupo encontrado
                                </SelectItem>
                              )}
                            </SelectContent>
                          </Select>
                          {repertoireGroups.length === 0 && (
                            <p className="text-sm text-amber-600 mt-1">
                              Você precisa criar um grupo de repertório primeiro. 
                              <br />
                              Acesse o Dashboard do Usuário para criar grupos.
                            </p>
                          )}
                        </div>

                        <Button 
                          onClick={addToRepertoire}
                          disabled={addingToRepertoire}
                          className="w-full"
                        >
                          {addingToRepertoire ? 'Adicionando...' : 'Adicionar ao Repertório'}
                        </Button>
                      </div>
                    </DialogContent>
                  </Dialog>
                )}
              </div>
            </div>
            </Card>
          ))}
        </div>

        {uniqueSongs.length === 0 && (
          <div className="text-center py-12">
            <Music className="h-16 w-16 text-gray-300 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              Nenhuma música encontrada
            </h3>
            <p className="text-gray-600">
              Tente ajustar os filtros ou termos de busca.
            </p>
          </div>
        )}
      </div>
    </div>
  );
}