"use client"

import React, { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"
import { Card, CardContent } from "@/components/ui/card"
import { X, Settings, Plus, Edit, Trash2 } from "lucide-react"

interface Category {
  id: number
  name: string
  created_at: string
}

interface CategoryManagementModalProps {
  isOpen: boolean
  onClose: () => void
}

export default function CategoryManagementModal({ isOpen, onClose }: CategoryManagementModalProps) {
  const [categories, setCategories] = useState<Category[]>([])
  const [isLoading, setIsLoading] = useState(false)
  const [newCategoryName, setNewCategoryName] = useState("")
  const [editingCategory, setEditingCategory] = useState<Category | null>(null)
  const [editCategoryName, setEditCategoryName] = useState("")
  const [deletingCategory, setDeletingCategory] = useState<Category | null>(null)
  const [isAddingCategory, setIsAddingCategory] = useState(false)

  useEffect(() => {
    if (isOpen) {
      loadCategories()
    }
  }, [isOpen])

  const loadCategories = async () => {
    setIsLoading(true)
    try {
      const response = await fetch("/api/categories")
      if (response.ok) {
        const data = await response.json()
        setCategories(data)
      }
    } catch (error) {
      console.error("Erro ao carregar categorias:", error)
    } finally {
      setIsLoading(false)
    }
  }

  const handleAddCategory = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!newCategoryName.trim()) return

    try {
      const response = await fetch("/api/categories", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ name: newCategoryName.trim() }),
      })

      if (response.ok) {
        setNewCategoryName("")
        setIsAddingCategory(false)
        loadCategories()
      } else {
        const error = await response.json()
        alert(`Erro ao criar categoria: ${error.error}`)
      }
    } catch (error) {
      console.error("Erro ao criar categoria:", error)
      alert("Erro ao criar categoria")
    }
  }

  const handleEditCategory = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!editingCategory || !editCategoryName.trim()) return

    try {
      const response = await fetch(`/api/categories/${editingCategory.id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ name: editCategoryName.trim() }),
      })

      if (response.ok) {
        setEditingCategory(null)
        setEditCategoryName("")
        loadCategories()
      } else {
        const error = await response.json()
        alert(`Erro ao editar categoria: ${error.error}`)
      }
    } catch (error) {
      console.error("Erro ao editar categoria:", error)
      alert("Erro ao editar categoria")
    }
  }

  const handleDeleteCategory = async () => {
    if (!deletingCategory) return

    try {
      const response = await fetch(`/api/categories/${deletingCategory.id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        setDeletingCategory(null)
        loadCategories()
      } else {
        const error = await response.json()
        alert(`Erro ao deletar categoria: ${error.error}`)
      }
    } catch (error) {
      console.error("Erro ao deletar categoria:", error)
      alert("Erro ao deletar categoria")
    }
  }

  const startEdit = (category: Category) => {
    setEditingCategory(category)
    setEditCategoryName(category.name)
  }

  const cancelEdit = () => {
    setEditingCategory(null)
    setEditCategoryName("")
  }

  return (
    <>
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-2">
                <Settings className="h-5 w-5 text-purple-600" />
                <DialogTitle>Gerenciar Categorias</DialogTitle>
              </div>
              <Button
                variant="ghost"
                size="sm"
                onClick={onClose}
                className="h-8 w-8 p-0"
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            <DialogDescription>
              Crie, edite e organize as categorias para classificar suas músicas.
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            {/* Adicionar Nova Categoria */}
            <Card>
              <CardContent className="p-4">
                {!isAddingCategory ? (
                  <Button
                    onClick={() => setIsAddingCategory(true)}
                    className="w-full"
                    variant="outline"
                  >
                    <Plus className="h-4 w-4 mr-2" />
                    Adicionar Nova Categoria
                  </Button>
                ) : (
                  <form onSubmit={handleAddCategory} className="space-y-3">
                    <div>
                      <Label htmlFor="new-category">Nome da Categoria</Label>
                      <Input
                        id="new-category"
                        value={newCategoryName}
                        onChange={(e) => setNewCategoryName(e.target.value)}
                        placeholder="Digite o nome da categoria"
                        required
                        autoFocus
                      />
                    </div>
                    <div className="flex space-x-2">
                      <Button type="submit" size="sm">
                        Criar
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setIsAddingCategory(false)
                          setNewCategoryName("")
                        }}
                      >
                        Cancelar
                      </Button>
                    </div>
                  </form>
                )}
              </CardContent>
            </Card>

            {/* Lista de Categorias */}
            <div className="space-y-2">
              <h3 className="text-lg font-semibold">Categorias Existentes</h3>
              {isLoading ? (
                <div className="text-center py-4">Carregando categorias...</div>
              ) : categories.length === 0 ? (
                <div className="text-center py-4 text-gray-500">
                  Nenhuma categoria encontrada
                </div>
              ) : (
                <div className="space-y-2">
                  {categories.map((category) => (
                    <Card key={category.id}>
                      <CardContent className="p-4">
                        {editingCategory?.id === category.id ? (
                          <form onSubmit={handleEditCategory} className="space-y-3">
                            <div>
                              <Label htmlFor="edit-category">Nome da Categoria</Label>
                              <Input
                                id="edit-category"
                                value={editCategoryName}
                                onChange={(e) => setEditCategoryName(e.target.value)}
                                required
                                autoFocus
                              />
                            </div>
                            <div className="flex space-x-2">
                              <Button type="submit" size="sm">
                                Salvar
                              </Button>
                              <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={cancelEdit}
                              >
                                Cancelar
                              </Button>
                            </div>
                          </form>
                        ) : (
                          <div className="flex items-center justify-between">
                            <div>
                              <h4 className="font-medium">{category.name}</h4>
                              <p className="text-sm text-gray-500">
                                Criada em: {new Date(category.created_at).toLocaleDateString('pt-BR')}
                              </p>
                            </div>
                            <div className="flex space-x-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => startEdit(category)}
                              >
                                <Edit className="h-4 w-4" />
                              </Button>
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setDeletingCategory(category)}
                                className="text-red-600 hover:text-red-700"
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          </div>
                        )}
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Dialog de Confirmação de Exclusão */}
      <AlertDialog open={!!deletingCategory} onOpenChange={() => setDeletingCategory(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Confirmar Exclusão</AlertDialogTitle>
            <AlertDialogDescription>
              Tem certeza que deseja deletar a categoria "{deletingCategory?.name}"?
              Esta ação não pode ser desfeita.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteCategory}
              className="bg-red-600 hover:bg-red-700"
            >
              Deletar
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  )
}