"use client"

import { useState } from "react"
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Play, Pause, SkipBack, SkipForward, Volume2, X } from "lucide-react"
import { Slider } from "@/components/ui/slider"

interface Song {
  id: number
  title: string
  artist: string
  youtubeUrl: string
}

interface MusicPlayerProps {
  isOpen: boolean
  onClose: () => void
  songs: Song[]
  currentSongIndex: number
  onSongChange: (index: number) => void
}

export default function MusicPlayer({ isOpen, onClose, songs, currentSongIndex, onSongChange }: MusicPlayerProps) {
  const [isPlaying, setIsPlaying] = useState(false)
  const [volume, setVolume] = useState([50])
  const [currentTime, setCurrentTime] = useState(0)
  const [duration, setDuration] = useState(0)

  const currentSong = songs[currentSongIndex]

  const getYouTubeVideoId = (url: string) => {
    if (!url || typeof url !== "string") {
      return null
    }

    // Diferentes padrões de URL do YouTube
    const patterns = [
      /(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/|youtube\.com\/v\/)([^#&?]*)/,
      /youtube\.com\/watch\?.*v=([^#&?]*)/,
      /youtu\.be\/([^#&?]*)/,
      /youtube\.com\/embed\/([^#&?]*)/,
      /youtube\.com\/v\/([^#&?]*)/,
    ]

    for (const pattern of patterns) {
      const match = url.match(pattern)
      if (match && match[1] && match[1].length === 11) {
        return match[1]
      }
    }

    return null
  }

  const handlePlayPause = () => {
    setIsPlaying(!isPlaying)
  }

  const handlePrevious = () => {
    if (currentSongIndex > 0) {
      onSongChange(currentSongIndex - 1)
    }
  }

  const handleNext = () => {
    if (currentSongIndex < songs.length - 1) {
      onSongChange(currentSongIndex + 1)
    }
  }

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60)
    const secs = Math.floor(seconds % 60)
    return `${mins}:${secs.toString().padStart(2, "0")}`
  }

  if (!currentSong) return null

  const videoId = getYouTubeVideoId(currentSong.youtubeUrl)

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl w-full">
        <DialogHeader>
          <div className="flex items-center justify-between">
            <DialogTitle className="text-xl font-bold">
              {currentSong.title} - {currentSong.artist}
            </DialogTitle>
            <Button variant="ghost" size="sm" onClick={onClose}>
              <X className="h-4 w-4" />
            </Button>
          </div>
          <DialogDescription>
            Player de música com controles de reprodução e visualização do vídeo do YouTube.
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* YouTube Player */}
          <div className="aspect-video bg-black rounded-lg overflow-hidden">
            {videoId ? (
              <iframe
                width="100%"
                height="100%"
                src={`https://www.youtube.com/embed/${videoId}?autoplay=${isPlaying ? 1 : 0}&controls=1&modestbranding=1&rel=0`}
                title={`${currentSong.title} - ${currentSong.artist}`}
                frameBorder="0"
                allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                allowFullScreen
              />
            ) : (
              <div className="flex items-center justify-center h-full text-white">
                <p>URL do YouTube inválida</p>
              </div>
            )}
          </div>

          {/* Player Controls */}
          <div className="space-y-4">
            {/* Progress Bar */}
            <div className="space-y-2">
              <Slider
                value={[currentTime]}
                max={duration}
                step={1}
                className="w-full"
                onValueChange={(value) => setCurrentTime(value[0])}
              />
              <div className="flex justify-between text-sm text-gray-500">
                <span>{formatTime(currentTime)}</span>
                <span>{formatTime(duration)}</span>
              </div>
            </div>

            {/* Control Buttons */}
            <div className="flex items-center justify-center space-x-4">
              <Button variant="outline" size="sm" onClick={handlePrevious} disabled={currentSongIndex === 0}>
                <SkipBack className="h-4 w-4" />
              </Button>

              <Button size="lg" onClick={handlePlayPause} className="rounded-full w-12 h-12">
                {isPlaying ? <Pause className="h-6 w-6" /> : <Play className="h-6 w-6" />}
              </Button>

              <Button variant="outline" size="sm" onClick={handleNext} disabled={currentSongIndex === songs.length - 1}>
                <SkipForward className="h-4 w-4" />
              </Button>
            </div>

            {/* Volume Control */}
            <div className="flex items-center space-x-2 max-w-xs mx-auto">
              <Volume2 className="h-4 w-4" />
              <Slider value={volume} max={100} step={1} className="flex-1" onValueChange={setVolume} />
              <span className="text-sm text-gray-500 w-8">{volume[0]}</span>
            </div>
          </div>

          {/* Playlist */}
          {songs.length > 1 && (
            <div className="space-y-2">
              <h3 className="font-semibold text-lg">Playlist</h3>
              <div className="max-h-40 overflow-y-auto space-y-1">
                {songs.map((song, index) => (
                  <div
                    key={song.id}
                    className={`flex items-center justify-between p-2 rounded cursor-pointer transition-colors ${
                      index === currentSongIndex ? "bg-purple-100 text-purple-900" : "hover:bg-gray-100"
                    }`}
                    onClick={() => onSongChange(index)}
                  >
                    <div className="flex-1 min-w-0">
                      <p className="font-medium text-sm truncate">{song.title}</p>
                      <p className="text-xs text-gray-500 truncate">{song.artist}</p>
                    </div>
                    {index === currentSongIndex && (
                      <div className="ml-2">
                        {isPlaying ? (
                          <Pause className="h-4 w-4 text-purple-600" />
                        ) : (
                          <Play className="h-4 w-4 text-purple-600" />
                        )}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  )
}
