-- Criar tabela de categorias de músicas
CREATE TABLE IF NOT EXISTS music_categories (
    id SERIAL PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    description TEXT,
    color VARCHAR(7) DEFAULT '#3B82F6', -- Cor hexadecimal para identificação visual
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Criar função para atualizar updated_at automaticamente
CREATE OR REPLACE FUNCTION update_music_categories_updated_at()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = CURRENT_TIMESTAMP;
    RETURN NEW;
END;
$$ language 'plpgsql';

-- Remover trigger existente se houver
DROP TRIGGER IF EXISTS update_music_categories_updated_at ON music_categories;

-- Criar trigger para atualizar updated_at (apenas se a tabela existir)
DO $$
BEGIN
    IF EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'music_categories') THEN
        CREATE TRIGGER update_music_categories_updated_at
            BEFORE UPDATE ON music_categories
            FOR EACH ROW
            EXECUTE FUNCTION update_music_categories_updated_at();
    END IF;
END $$;

-- Inserir categorias padrão
INSERT INTO music_categories (name, description, color) VALUES
('Rock', 'Músicas de rock e suas variações', '#EF4444'),
('Pop', 'Música popular contemporânea', '#F59E0B'),
('Jazz', 'Jazz e suas variações', '#8B5CF6'),
('Blues', 'Blues tradicional e moderno', '#06B6D4'),
('Country', 'Música country e sertaneja', '#10B981'),
('Folk', 'Música folk e tradicional', '#F97316'),
('Gospel', 'Música gospel e religiosa', '#6366F1'),
('MPB', 'Música Popular Brasileira', '#EC4899'),
('Samba', 'Samba e pagode', '#84CC16'),
('Bossa Nova', 'Bossa nova e jazz brasileiro', '#14B8A6'),
('Forró', 'Forró e música nordestina', '#F59E0B'),
('Reggae', 'Reggae e música jamaicana', '#22C55E'),
('Eletrônica', 'Música eletrônica e dance', '#A855F7'),
('Clássica', 'Música clássica e erudita', '#6B7280'),
('Instrumental', 'Música instrumental', '#9CA3AF'),
('Outros', 'Outras categorias não listadas', '#64748B')
ON CONFLICT (name) DO NOTHING;

-- Atualizar tabela songs para incluir categoria
ALTER TABLE songs 
ADD COLUMN IF NOT EXISTS category_id INTEGER REFERENCES music_categories(id) DEFAULT 16; -- Default para "Outros"

-- Criar índices para melhor performance
CREATE INDEX IF NOT EXISTS idx_songs_category ON songs(category_id);
CREATE INDEX IF NOT EXISTS idx_music_categories_name ON music_categories(name);